<?php
/**
 * Installation Check Script
 * Run this file to check if your environment is ready
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

$checks = [];
$allPassed = true;

// Check PHP Version
$phpVersion = phpversion();
$checks['php_version'] = [
    'name' => 'PHP Version',
    'required' => '7.4+',
    'current' => $phpVersion,
    'passed' => version_compare($phpVersion, '7.4.0', '>=')
];
if (!$checks['php_version']['passed']) $allPassed = false;

// Check MySQL Extension
$checks['mysqli'] = [
    'name' => 'MySQLi Extension',
    'required' => 'Enabled',
    'current' => extension_loaded('mysqli') ? 'Enabled' : 'Disabled',
    'passed' => extension_loaded('mysqli')
];
if (!$checks['mysqli']['passed']) $allPassed = false;

// Check PDO Extension
$checks['pdo'] = [
    'name' => 'PDO Extension',
    'required' => 'Enabled',
    'current' => extension_loaded('pdo') ? 'Enabled' : 'Disabled',
    'passed' => extension_loaded('pdo')
];
if (!$checks['pdo']['passed']) $allPassed = false;

// Check required files
$requiredFiles = [
    'config/database.php',
    'config/auth.php',
    'includes/header.php',
    'includes/footer.php',
    'login.php',
    'dashboard.php'
];

$missingFiles = [];
foreach ($requiredFiles as $file) {
    if (!file_exists($file)) {
        $missingFiles[] = $file;
    }
}

$checks['files'] = [
    'name' => 'Required Files',
    'required' => 'All present',
    'current' => empty($missingFiles) ? 'All present' : 'Missing: ' . implode(', ', $missingFiles),
    'passed' => empty($missingFiles)
];
if (!$checks['files']['passed']) $allPassed = false;

// Check database connection
$dbConnected = false;
$dbError = '';
if (file_exists('config/database.php')) {
    require_once 'config/database.php';
    try {
        $conn = getDBConnection();
        $dbConnected = true;
        $conn->close();
    } catch (Exception $e) {
        $dbError = $e->getMessage();
    }
}

$checks['database'] = [
    'name' => 'Database Connection',
    'required' => 'Connected',
    'current' => $dbConnected ? 'Connected' : 'Failed: ' . $dbError,
    'passed' => $dbConnected
];
if (!$checks['database']['passed']) $allPassed = false;

// Check write permissions
$writableDirs = [];
$checks['permissions'] = [
    'name' => 'Write Permissions',
    'required' => 'Writable',
    'current' => 'OK',
    'passed' => true
];
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Installation Check - SIDAPBUN</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #f0fdf4 0%, #fefce8 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        .container {
            background: white;
            border-radius: 12px;
            padding: 40px;
            max-width: 800px;
            width: 100%;
            box-shadow: 0 10px 40px rgba(0, 0, 0, 0.1);
        }
        h1 {
            color: #1f2937;
            margin-bottom: 8px;
        }
        .subtitle {
            color: #6b7280;
            margin-bottom: 32px;
        }
        .check-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 16px;
            border-radius: 8px;
            margin-bottom: 12px;
            background: #f9fafb;
        }
        .check-item.passed {
            background: #d1fae5;
        }
        .check-item.failed {
            background: #fee2e2;
        }
        .check-name {
            font-weight: 600;
            color: #1f2937;
        }
        .check-details {
            font-size: 14px;
            color: #6b7280;
        }
        .status {
            font-weight: bold;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
        }
        .status.passed {
            background: #10b981;
            color: white;
        }
        .status.failed {
            background: #ef4444;
            color: white;
        }
        .summary {
            margin-top: 32px;
            padding: 20px;
            border-radius: 8px;
            text-align: center;
        }
        .summary.success {
            background: #d1fae5;
            color: #065f46;
        }
        .summary.error {
            background: #fee2e2;
            color: #991b1b;
        }
        .btn {
            display: inline-block;
            padding: 12px 24px;
            background: #22c55e;
            color: white;
            text-decoration: none;
            border-radius: 8px;
            margin-top: 24px;
            font-weight: 500;
        }
        .btn:hover {
            background: #16a34a;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>Installation Check</h1>
        <p class="subtitle">SIDAPBUN - Sistem Input Data Produksi</p>
        
        <?php foreach ($checks as $check): ?>
            <div class="check-item <?php echo $check['passed'] ? 'passed' : 'failed'; ?>">
                <div>
                    <div class="check-name"><?php echo htmlspecialchars($check['name']); ?></div>
                    <div class="check-details">
                        Required: <?php echo htmlspecialchars($check['required']); ?> | 
                        Current: <?php echo htmlspecialchars($check['current']); ?>
                    </div>
                </div>
                <span class="status <?php echo $check['passed'] ? 'passed' : 'failed'; ?>">
                    <?php echo $check['passed'] ? 'PASS' : 'FAIL'; ?>
                </span>
            </div>
        <?php endforeach; ?>
        
        <div class="summary <?php echo $allPassed ? 'success' : 'error'; ?>">
            <h2><?php echo $allPassed ? '✓ All Checks Passed!' : '✗ Some Checks Failed'; ?></h2>
            <p>
                <?php if ($allPassed): ?>
                    Your environment is ready! You can now use the application.
                <?php else: ?>
                    Please fix the issues above before using the application.
                <?php endif; ?>
            </p>
        </div>
        
        <?php if ($allPassed): ?>
            <div style="text-align: center;">
                <a href="login.php" class="btn">Go to Login</a>
            </div>
        <?php endif; ?>
    </div>
</body>
</html>

